function [punc_data] = punc_metrics(varargin)
%This function is a script that helps the user perform a puncta
%wise colocaliztion analysis.  The user will enter a minimum of two x y z
%coordinate lists and the program will spit out a structure of distances
%and vector metrics.
%Synatax:   [punc_data] = punc_metrics('nhood',20); 
%Input:     This function does not really require any inputs.  However if
%           you do not want default values then..
%           'nhood' = the search neighborhood around each vertex for its
%                   nearest neighbor.  Default = 10
%           'center' = value of 1 or 0, 0 = off, 1 = on.  When center is on
%                   calculations for vectors will be made towards a point
%                   between the two defined points, and there will be an
%                   extra dataset for the distance meteric which will
%                   represent the center between the two selected points.
%                   Default = 0.
%           'vector' = value of 1 or 0, 0 = off, 1 = on.  When on the
%                   vector metric will be attempted for the data input.
%                   Default = 1.
%           'x' = the scale, as a ratio or xyz, of 1 unit in the x axis.
%                 Default = 1
%           'y' = the scale, as a ratio or xyz, of 1 unit in the y axis.
%                 Default = 1
%           'z' = the scale, as a ratio or xyz, of 1 unit in the z axis.
%                 Default = 1
%           'nrst_only' = Calculate vector using the nearst points as well. 
%                   otherwise, vector will be calculated using the full
%                   list of vertices in the nhood volumn.  Default = 0/off.
%                   1 = on.
%Output:    Punc_data = a structured array with the data for puncta data
%           The structure of the data set is this:
%           Distance Metric:
%             punc_data is a structure of depth x, where x is the number of
%             imported vertices lists.
%             punc_data(x).distance = An x by 1 array where x matches the
%               number of verticies in that particular list.
%             punc_data(x).vertices = An x by 3 array of vertices that was
%               used for generating the distances.
%             punc_data(x).closest_pt = An y by 1 by x cell array of
%               verticies [a 3 b] where y = the number of verices used to 
%               generate the distances. x = the number of comparision 
%               lists. a = the number of closest points to central pivot.  
%               a is >1 only if there are two points equaldistant from the 
%               pivot.
%             punc_data(x).verti_match = An y by 1 by x cell array of
%               vertices [a 3 b] that mirrors the .closest_pt cell array.
%             punc_data(x).dist_verti = An y by 1 by x cell array of
%               distances [a 1 b] that mirrors the .closest_pt cell array 
%               and the .verti_match cell array.
%             punc_data(x).short_list = An y by 1 by x cell array of
%               verticies arrays[a 3 b] nhood away from the central pivot.
%               x matches the x in .verti_match...etc.
%           Vector Metric:
%             punc_data(i).angles_xy = An x by y by z cell array of a by 1
%               cell array of a 1 by b array.  x is the number of start 
%               points for a particular channels, which is basically, ch1 =
%               start point, ch2 = terminal point, and x = total # of start
%               points.  You can have multiple terminals points per start 
%               point. y is a channel not ch1 or ch2, with max dim = total 
%               channel #.  z = number of channels in the dataset.  a = 
%               number of terminal points per start point, in ch2, which 
%               together defines the base vector. b = number of terminal
%               points (ch3) per base vector (ch2 terminal points) per 
%               start point. i = ch1.  The array contains data describing
%               the angle of a vector away for a base vector in the xy
%               plane.
%             punc_data(i).angles_xz = An x by y by z cell array of a by 1
%               cell array of a 1 by b array.  i = ch1.  The array contains 
%               data describing the angle of a vector away for a base 
%               vector in the xz plane.
%             punc_data(i).angles_xyz = An x by y by z cell array of a by 1
%               cell array of a 1 by b array.  i = ch1.  The array contains 
%               data describing the angle of a vector away for a base 
%               vector in the xyz space.  The angle essentially treates the
%               two vectors as if they are in some arbitrary plane.
%             punc_data(i).angles_pos_xy = An x by y by z cell array of a by 1
%               cell array of a 1 by b array.  i = ch1.  The array contains 
%               data describing the sign of the angles desribed in the xy 
%               array.
%             punc_data(i).angles_pos_xz = An x by y by z cell array of a by 1
%               cell array of a 1 by b array.  i = ch1.  The array contains 
%               data describing the sign of the angles desribed in the xz 
%               array.

[nhood,vector,center,x_scale,y_scale,z_scale,nrst_only] = parse(varargin);     %Parse the input

%import vertex files
%open the first vertex file. Prompt first
prompt_box('title','Open Vertex Lists','prompt1','Select the vertex lists you want to compare.','position','center');
open_data = 0;      %initialize open files switch
vertices = {};      %initialize vertex array
filenames = {};
pathnames = {};  
while open_data==0
    %this line helps get around the 100 file limitation
    setappdata(0,'UseNativeSystemDialogs',false)
    %get files. Note: currently the filterindex is the filter selected for the file selection.
    [filename_tmp,pathname_tmp,filterindex] = uigetfile2({'*.xlsx','Excel 2007 files (*.xlsx)';...
        '*.xls','Excel files (*.xls)';'*.txt','Text files (*.csv)';'*.mat','Mat files (*.mat)';},...
        'Open Stack','Multiselect','on');
    setappdata(0,'UseNativeSystemDialogs',true)
    %you must load at least two data sets
    if filterindex~=4       %mat files are treated differently
        if ischar(filename_tmp)
            error('You must load at least two data sets.')
        end
    elseif ischar(filename_tmp)     %if is mat file make the single file a 1x1 cell array so it will work.
        filename_tmp = {filename_tmp};
    end
    %Now open the data sets according to the filter used.
    switch filterindex
        case {1,2}              %excel file
            for i = 1:size(filename_tmp,2)
                verti_tmp{i} = single(xlsread([pathname_tmp,filename_tmp{1,i}]));
            end
            vertices = cat(2,vertices,verti_tmp);       %join all the opened data together
        case 3                  %text file - Default coma delimited
            for i = 1:size(filename_tmp,2)
                verti_tmp{i} = single(dlmread([pathname_tmp,filename_tmp{1,i}]));
            end
            vertices = cat(2,vertices,verti_tmp);       %join all the opened data together
        otherwise               %MAT FILES      - Note Note supported right now.
            for i = 1:size(filename_tmp,2)
                load([pathname_tmp,filename_tmp{1,i}]);
            end
    end
    filenames = cat(2,filenames,filename_tmp);
    pathnames = cat(2,pathnames,pathname_tmp);
    open_data = yes_no_box('title','Are you done selecting data?','caption1','Select yes is you want to move on.',...
        'caption2','Select no if you want to import another dataset.','position','center');
end
%clean up
clear verti_tmp filename_tmp pathname_tmp

%store the file properties
punc_data.filenames = filenames;
punc_data.pathnames = pathnames;
if size(filenames,2)<3 && vector       %needs at least 3 data sets for vector to work
    vector = 0;
end
punc_data.vector = vector;              %store in the data structure whether vector calculations are made

%begin parallelization
parpool      %initiate processes

%now lets calculate the distance metric
[punc_data] = distance_cal(filenames,vertices,punc_data,nhood,x_scale,y_scale,z_scale,nrst_only);
%clean up & consolidate
clear vertices

%Now calculate the vector, this part uses the first two data sets to create
%a set vectors that will be used as the base for the vector calculations of
%all the other datasets.  The results are sets of vectors that points away
%from this new central axis that represent the vector way from central
%vector.
%first prepare the verticies data points from the punc_data structure for
%vector analysis.
if vector       %calculate vector or not
    if nrst_only            %calculate using the closeset point only if selected...for now, until faster computers.
        for i = 1:size(filenames,2)
            ori_data{i,1} = punc_data(i).verti_match;  %the center point from which all vectors are drawn
            termini_data{i,1} = punc_data(i).closest_pt;     %the termination point of the base & comparision vector
            %second_pt_data{i,1} = punc_data(i).closest_pt;  %get the termini for the second vector
        end
        
        %now calculate the vector info
        [punc_tmp] = vector_cal(ori_data,termini_data,filenames,vector);
        
        %store angles for output
        for i = 1:size(filenames,2)
            punc_data(i).angles_xy = punc_tmp(i).angles_xy;
            punc_data(i).angles_xz = punc_tmp(i).angles_xz;
            punc_data(i).angles_xyz = punc_tmp(i).angles_xyz;
            punc_data(i).angles_pos_xy = punc_tmp(i).angles_pos_xy;
            punc_data(i).angles_pos_xz = punc_tmp(i).angles_pos_xz;
        end
        
        punc_data(1).nrst_only = 1;     %nearest available
        
        %clear datafields for next run
        clear ori_data termini_data punc_tmp
    else            %let programes know that there is no closest point data structure.
        punc_data(1).nrst_only = 0;
    end
    
    %Now prepare the verticies data points from the punc_data structure for
    %vector analysis of the short list data structures.
    for i = 1:size(filenames,2)
        ori_data{i,1} = punc_data(i).sl_verti_match;  %the center point from which all vectors are drawn
        termini_data{i,1} = punc_data(i).short_list;     %the termination point of the base & comparision vector
    end
    
    %now calculate the vector info
    [punc_tmp] = vector_cal(ori_data,termini_data,filenames,vector);
    
    %store short list angles for output
    for i = 1:size(filenames,2)
        punc_data(i).sl_angles_xy = punc_tmp(i).angles_xy;
        punc_data(i).sl_angles_xz = punc_tmp(i).angles_xz;
        punc_data(i).sl_angles_xyz = punc_tmp(i).angles_xyz;
        punc_data(i).sl_angles_pos_xy = punc_tmp(i).angles_pos_xy;
        punc_data(i).sl_angles_pos_xz = punc_tmp(i).angles_pos_xz;
    end
end

%closing time
delete(gcp('nocreate'))

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [nhood,vector,center,x_scale,y_scale,z_scale,nrst_only] = parse(input)

nhood = 10;  %Default Initialized.
center = 0;  %Center is off at default
vector = 1;  %vector is on by default
x_scale = 1;    %base scale of a unit in the x axis
y_scale = 1;    %base scale of a unit in the y axis
z_scale = 1;    %base scale of a unit in the z axis
nrst_only = 0;  %Nearest off by default

%Parse the input
if ~isempty(input)
    for i = 1:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'nhood'
                    nhood = input{1,i+1};
                case 'center'
                    center = input{1,i+1};
                case 'vector'
                    vector = input{1,i+1};
                case 'x'
                    x_scale = input{1,i+1};
                case 'y'
                    y_scale = input{1,i+1};
                case 'z'
                    z_scale = input{1,i+1};
                case 'nrst_only'
                    nrst_only = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        end
    end
end

%--------------------------------------------------------------------------
%subfunction to calculate the distance metric.
function [punc_data] = distance_cal(filenames,vertices,punc_data,nhood,x_scale,y_scale,z_scale,nrst_only)

%If not subtraction list is presented, do not subtract any point from the
%data structure.
if nargin==4
    sub_list = 0;
end

%parallelize this puppy
%matlabpool      %initiate processes

%now lets calculate the distance metric
%h = waitbar(0,['Data Sets Analyzed: ',filenames{1}],'position',[20 300 275 50]);    %initialize progress bar.
parfor i = 1:size(filenames,2)      %calculate for each data set
    %seperate out the brightness value
    verti_tmp = vertices{1,i}(:,1:3);     %pull out one data set to work with
    bright_tmp = single(vertices{1,i}(:,4));        %brightness data for each vertices
    verti_tmp = single(verti_tmp.*repmat([x_scale y_scale z_scale],size(verti_tmp,1),1));   %scale the vertices to the proper unit
    %preload cell arrays
    dist_tmp = zeros(size(verti_tmp,1),1,size(filenames,2));       %shortest distances
    verti_match = cell(size(verti_tmp,1),1,size(filenames,2)); %the matching original verticies for the closeset points.
    v_match_lum = cell(size(verti_tmp,1),1,size(filenames,2)); %the matching luminance values
    dist_verti_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %the matching distance matrix to the closest point list. A cell array as well.
    closest_tmp = cell(size(verti_tmp,1),1,size(filenames,2));  %the closest pivot.  A cell array.
    closest_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));  %the matching lumninace data.  A cell array
    short_tmp = cell(size(verti_tmp,1),1,size(filenames,2));    %The short list.  A cell Array.
    short_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));    %The short list luminance.  A cell array.
    sl_dist_tmp = cell(size(verti_tmp,1),1,size(filenames,2));     %The short list distances.  A cell Array.
    sl_verti_match_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %The matching verticies for the short list points.
    sl_v_match_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %The matching luminance data.
    %start progress bar
    %h2 = waitbar(0,['Comparing Data Set: ',filenames{1}],'position',[20 200 275 50]);    %initialize progress bar.
    for j = 1:size(filenames,2)      %get the comparision set
        if j~=i                     %skip if it is the base set
            %h3 = waitbar(0,'Vertex Distance Analyzed: 0');    %initialize progress bar.
            for k = 1:size(verti_tmp,1)     %iterate through each vertex of the set
                display(['distance ',filenames{1,i},' ',filenames{1,j},' ',num2str(i),'_',num2str(j),'_',num2str(k)]);
                verti_co_tmp = vertices{1,j}(:,1:3).*repmat([x_scale y_scale z_scale],size(vertices{1,j},1),1);   %scale the vertices to the proper unit
                bright_co_tmp = vertices{1,j}(:,4);     %matching brightness data for comparision channel
                [short_list short_list_lum] = find_neighbors(verti_tmp(k,:),verti_co_tmp,nhood,bright_co_tmp);    %get a short list of local pivots
                if ~isempty(short_list)           %make sure there is something in the neighborhood.
                    curr_verti = repmat(verti_tmp(k,:),size(short_list,1),1);         %replicate the current vertex to the size of the short list.
                    verti_dist = single(dddist(curr_verti,short_list));     %calculate the distance of the current vertex from all points in the short list.
                    [x,y] = find(verti_dist==absmin(verti_dist));   %find the position of the closest point
                    closest_tmp{k,1,j} = short_list(x,:);           %create a cell array of all the closest points, just in case of multiples.
                    closest_lum_tmp{k,1,j} = short_list_lum(x,:);   %create a matching cell array of luminance
                    verti_match{k,1,j} = repmat(verti_tmp(k,:),size(short_list(x,:),1),1);    %create a matching original verticies for the closets pts.
                    v_match_lum{k,1,j} = repmat(bright_tmp(k,:),size(short_list(x,:),1),1);
                    dist_verti_tmp{k,1,j} = verti_dist(x,:);  %create a matching set of distances with the vertices.
                    dist_tmp(k,1,j) = verti_dist(x(1,1),:);              %create a columnar vector of shortest distances
                    short_tmp{k,1,j} = short_list;    %output the short list as well.
                    short_lum_tmp{k,1,j} = short_list_lum;  %output the luminances of the short list as well
                    sl_dist_tmp{k,1,j} = verti_dist;  %output the distances for all points in the short list.
                    sl_verti_match_tmp{k,1,j} = repmat(verti_tmp(k,:),size(short_list,1),1);  %create a matching vertices for the short list
                    sl_v_match_lum_tmp{k,1,j} = repmat(bright_tmp(k,:),size(short_list,1),1);  %create a matching luminance for the short list
                else                        %there is no one around me (sad)
                    closest_tmp{k,1,j} = single([NaN NaN NaN]);           %not a number for easy extraction.
                    closest_lum_tmp{k,1,j} = single(NaN);
                    verti_match{k,1,j} = verti_tmp(k,:);    
                    v_match_lum{k,1,j} = bright_tmp(k,:);
                    dist_verti_tmp{k,1,j} = single(NaN);  
                    dist_tmp(k,1,j) = single(NaN);
                    short_tmp{k,1,j} = single([NaN NaN NaN]);    %output the short list as well.
                    short_lum_tmp{k,1,j} = single(NaN);
                    sl_dist_tmp{k,1,j} = single(NaN);   %output the distance for short list.
                    sl_verti_match_tmp{k,1,j} = verti_tmp(k,:);  %all by yourself
                    sl_v_match_lum_tmp{k,1,j} = bright_tmp(k,:);
                end
                %waitbar(k/size(verti_tmp,1),h3,['Vertex Distance Analyzed: ',num2str(k)]);   %update progress
            end
            %close(h3);   %close progress bar
            %waitbar(j/size(filenames,2),h2,['Comparing Data Set: ',filenames{j}]);   %update progress
        end
    end
    %store the data. The structure is that the distance is a matrix that
    %corresponds with the vertices list in the rows and with each z step a
    %calculation against another dataset, the self to self comparision data
    %set is all 0.  So 4 datasets, will be a matrix of (x,1,4).  The
    %vertices is just the verticies used to calculate the distance.  The
    %closest_pt dataset is a cell array that matches the verticies list in
    %rows and with each z step representing another dataset, the cell array
    %is used for the instances where there are multiple closest points.
    punc_data(i).distance = dist_tmp;       %shortest distances
    punc_data(i).vertices = verti_tmp;      %the vertices 
    punc_data(i).verti_match = verti_match; %the matching original verticies for the closeset points.
    punc_data(i).v_match_lum = v_match_lum; %the matching luminance values
    punc_data(i).dist_verti = dist_verti_tmp;   %the matching distance matrix to the closest point list. A cell array as well.
    punc_data(i).closest_pt = closest_tmp;  %the closest pivot.  A cell array.
    punc_data(i).closest_pt_lum = closest_lum_tmp;  %the matching lumninace data.  A cell array
    punc_data(i).short_list = short_tmp;    %The short list.  A cell Array.
    punc_data(i).short_list_lum = short_lum_tmp;    %The short list luminance.  A cell array.
    punc_data(i).sl_dist = sl_dist_tmp;     %The short list distances.  A cell Array.
    punc_data(i).sl_verti_match = sl_verti_match_tmp;   %The matching verticies for the short list points.
    punc_data(i).sl_v_match_lum = sl_v_match_lum_tmp;   %The matching luminance data.
    %clear data for reinitialization
    %clear verti_tmp verti_co_tmp dist_tmp closest_tmp verti_match dist_verti_tmp short_tmp sl_dist_tmp sl_verti_match_tmp bright_tmp bright_co_tmp closest_lum_tmp v_match_lum short_lum_tmp sl_v_match_lum_tmp
    verti_tmp = []; verti_co_tmp = []; dist_tmp = []; closest_tmp = []; verti_match = []; dist_verti_tmp = []; short_tmp = []; sl_dist_tmp = [];
    sl_verti_match_tmp = []; bright_tmp = []; bright_co_tmp = []; closest_lum_tmp = []; v_match_lum = []; short_lum_tmp = []; sl_v_match_lum_tmp = [];
    %update progress bars
    %close(h2);   %close progress bar
    %waitbar(i/size(filenames,2),h,['Data Sets Analyzed: ',filenames{i}]);   %update progress
end
%close(h);   %close progress bar
clear verti_tmp verti_co_tmp dist_tmp closest_tmp verti_match dist_verti_tmp short_tmp sl_dist_tmp sl_verti_match_tmp bright_tmp bright_co_tmp closest_lum_tmp v_match_lum short_lum_tmp sl_v_match_lum_tmp
%matlabpool close %close processes

%--------------------------------------------------------------------------
%subfunction to calculate the vector metric.
function [punc_tmp] = vector_cal(ori_data,termini_data,filenames,vector)
%In the following function, i = total of all start points in punc_data for
%a specific channel.  j = total of all terminal points from the start
%points in i; essentially, z steps in the closest_pt cell array, of that
%channel.  k = number of start points in i. l = second terminal point, of another 
%channel not j, that is used to calculate the angle; essentially, z step in
%the closest_pt cell array, not i or j.  m = number of terminal point (j) per 
%start point for channel i.

%parallelize this puppy
%matlabpool      %initiate processes

%h = waitbar(0,['Data Sets Analyzed: ',filenames{1}],'position',[20 300 275 50]);    %initialize progress bar.
parfor i = 1:size(filenames,2)      %calculate for each data set
    %preallocate cell arrays
    angles_xy = cell(size(ori_data{i,1},1),size(filenames,2),size(filenames,2));
    angles_xz = cell(size(ori_data{i,1},1),size(filenames,2),size(filenames,2));
    angles_xyz = cell(size(ori_data{i,1},1),size(filenames,2),size(filenames,2));
    angles_pos_xy = cell(size(ori_data{i,1},1),size(filenames,2),size(filenames,2));
    angles_pos_xz = cell(size(ori_data{i,1},1),size(filenames,2),size(filenames,2));
    %h2 = waitbar(0,['Comparing Data Set: ',filenames{1}],'position',[20 200 275 50]);    %initialize progress bar.
    for j = 1:size(filenames,2)  %step through each other data set
        if i~=j             %skip the self on self comparision
            %h3 = waitbar(0,'Vertex Vector Analyzed: 0');    %initialize progress bar.
            for k = 1:size(ori_data{i,1},1)   %step through the verticies or pivots of channel j
                display(['vector ',filenames{1,i},' ',filenames{1,j},' ',num2str(i),'_',num2str(j),'_',num2str(k)]);
                ori = ori_data{i,1}{k,:,j};  %the center point from which all vectors are drawn
                first_pt = termini_data{i,1}{k,:,j};     %the termination point of the base vector
                %calculate the base vector for the xy plane
                ori_xy = ori(:,1:2)';        %array of the xy position of the origin
                first_xy = first_pt(:,1:2)';        %array of the xy position of the termini of the base vector
                base_xy_dist = first_xy - ori_xy;    %array of the x & y distance apart
                base_xy_v = hypot(base_xy_dist(1,:),base_xy_dist(2,:));   %arry of vector distances for the base vector
                %calculate the base vector for the xz plane
                ori_xz = ori(:,1:2:3)';      %array of the xz position of the origin
                first_xz = first_pt(:,1:2:3)';      %array of the xz position of the termini of the base vector
                base_xz_dist = first_xz - ori_xz;    %array of the x & y distance apart
                base_xz_v = hypot(base_xz_dist(1,:),base_xz_dist(2,:));   %arry of vector distances for the base vector
                %calculate the base vector for the coplaner case
                base_xyz_dist = first_pt'-ori';       %array of x, y & z distances
                base_xyz_v = sqrt(sum(base_xyz_dist.^2));   %array of vector distances for the base vector
                for l = 1:size(filenames,2)    %step through the rest of dataset to find the second vector and the angle
                    if l~=j && l~=i                     %skip the original vector datasets
                        %preallocate data structures
                        angles_xy_tmp = cell(size(ori,1),1);       %store the angles in a cell array.
                        angles_xz_tmp = cell(size(ori,1),1);
                        angles_xyz_tmp = cell(size(ori,1),1);
                        angles_pos_xy_tmp = cell(size(ori,1),1);
                        angles_pos_xz_tmp = cell(size(ori,1),1);
                        for m = 1:size(ori,1)           %for each origin to first point vector test all second point vectors.
                            second_pt = termini_data{i,1}{k,:,l};   %get the termini for the second vector
                            second_xy = second_pt(:,1:2)';                      %Array of xy positions for the termini of the comparision vector
                            second_xz = second_pt(:,1:2:3)';                    %Array of xz positions for the termini of the comparision vector
                            ori_xy_tmp = repmat(ori_xy(:,m),1,size(second_xy,2));   %match the origin arrays with the second point arrays
                            ori_xz_tmp = repmat(ori_xz(:,m),1,size(second_xy,2));
                            first_xy_tmp = repmat(first_xy(:,m),1,size(second_xy,2));   %match the first arrays with the second point arrays
                            first_xz_tmp = repmat(first_xz(:,m),1,size(second_xy,2));
                            base_xy_dist_tmp = repmat(base_xy_dist(:,m),1,size(second_xy,2)); %match the dist arrays with the second point arrays
                            base_xz_dist_tmp = repmat(base_xz_dist(:,m),1,size(second_xy,2));
                            base_xyz_dist_tmp = repmat(base_xyz_dist(:,m),1,size(second_xy,2));
                            base_xy_v_tmp = repmat(base_xy_v(:,m),1,size(second_xy,2)); %match the vector arrays with the second point arrays
                            base_xz_v_tmp = repmat(base_xz_v(:,m),1,size(second_xy,2));
                            base_xyz_v_tmp = repmat(base_xyz_v(:,m),1,size(second_xy,2));
                            ori_tmp = repmat(ori(m,:),size(second_pt,1),1);
                            %calculate in xy plane
                            comp_xy_dist = second_xy - ori_xy_tmp;              %array of the x & y distances
                            comp_xy_v = hypot(comp_xy_dist(1,:),comp_xy_dist(2,:));   %array of vector distances for the comparision vector
                            dot_xy = dot(base_xy_dist_tmp,comp_xy_dist); %get the dot product of the two vectors
                            angles_xy_tmp{m,1} = (acos(dot_xy./(base_xy_v_tmp.*comp_xy_v))*180/pi)';  %calculate the angle.
                            %calculate whether the angle is left or right of the
                            %vector
                            [angles_pos_xy_tmp{m,1}] = angle_position(ori_xy_tmp,first_xy_tmp,second_xy)';
                            %calculate in xz plane
                            comp_xz_dist = second_xz - ori_xz_tmp;              %array of the x & z distances
                            comp_xz_v = hypot(comp_xz_dist(1,:),comp_xz_dist(2,:));   %array of vector distances for the comparision vector
                            dot_xz = dot(base_xz_dist_tmp,comp_xz_dist); %get the dot product of the two vectors
                            angles_xz_tmp{m,1} = (acos(dot_xz./(base_xz_v_tmp.*comp_xz_v))*180/pi)';  %calculate the angle.
                            %calculate whether the angle is left or right of the
                            %vector
                            [angles_pos_xz_tmp{m,1}] = angle_position(ori_xz_tmp,first_xz_tmp,second_xz)';
                            %calculate in the coplanar plane for the two vectors
                            comp_xyz_dist = second_pt' - ori_tmp';                  %arry of x, y and z distances
                            comp_xyz_v = sqrt(sum(comp_xyz_dist.^2));       %array of vector distances for the comparision vector
                            dot_xyz = dot(base_xyz_dist_tmp,comp_xyz_dist); %the dot product of the two vectors
                            angles_xyz_tmp{m,1} = (acos(dot_xyz./(base_xyz_v_tmp.*comp_xyz_v))*180/pi)';    %calculate the angle.
                            %clear up some strange stuff.  acos(24/24.000) is not imaginary.
                            if ~isreal(angles_xy_tmp{m,1})      %if not real
                                angles_xy_tmp{m,1} = real(angles_xy_tmp{m,1});      %take only the real component
                            end
                            if ~isreal(angles_xz_tmp{m,1})      %same as above for xz
                                angles_xz_tmp{m,1} = real(angles_xz_tmp{m,1});
                            end
                            if ~isreal(angles_xyz_tmp{m,1})     %same as above for xyz
                                angles_xyz_tmp{m,1} = real(angles_xyz_tmp{m,1});
                            end
                            %create a data structure for storing the data
                            angles_xy{k,l,j} = angles_xy_tmp;       %store the angles in a cell array.
                            angles_xz{k,l,j} = angles_xz_tmp;
                            angles_xyz{k,l,j} = angles_xyz_tmp;
                            angles_pos_xy{k,l,j} = angles_pos_xy_tmp;
                            angles_pos_xz{k,l,j} = angles_pos_xz_tmp;
                            
                            %clear all of the temporary data
                            if m==size(ori,1)   %clear only at the end of the cycle
                                %clear angles_xy_tmp angles_xz_tmp angles_xyz_tmp angles_pos_xy_tmp angles_pos_xz_tmp
                                angles_xy_tmp = []; angles_xz_tmp = []; angles_xyz_tmp = []; angles_pos_xy_tmp = []; angles_pos_xz_tmp = [];
                            end
                        end
                    end
                end
                %waitbar(k/size(ori_data{i,1},1),h3,['Vertex Vector Analyzed: ',num2str(k)]);   %update progress
            end
            
            %close(h3);   %close progress bar
            %waitbar(j/size(filenames,2),h2,['Comparing Data Set: ',filenames{j}]);   %update progress
        end
    end
    %store angles for output
    punc_tmp(i).angles_xy = angles_xy;
    punc_tmp(i).angles_xz = angles_xz;
    punc_tmp(i).angles_xyz = angles_xyz;
    punc_tmp(i).angles_pos_xy = angles_pos_xy;
    punc_tmp(i).angles_pos_xz = angles_pos_xz;
    
    %clear holder data arrays
    %clear angles_xy angles_xz angles_xyz angles_pos_xy angles_pos_xz
    angles_xy = []; angles_xz = []; angles_xyz = []; angles_pos_xy = []; angles_pos_xz = [];
    %close(h2);
    %waitbar(i/size(filenames,2),h,['Data Sets Analyzed: ',filenames{i}]);   %update progress
end
%close(h);

%clear holder data arrays
clear angles_xy angles_xz angles_xyz angles_pos_xy angles_pos_xz angles_xy_tmp angles_xz_tmp angles_xyz_tmp angles_pos_xy_tmp angles_pos_xz_tmp
%matlabpool close %close processes

%--------------------------------------------------------------------------
%subfunction to calculate the left of right handedness of the angle
function [a_pos] = angle_position(ori_mg,first_mg,second_mg)
%To determine whether a second vector is to left or right of a base vector
%you take the magnitudes, here the unit vector is 1 in cartesian xyz, of
%each component vector in two directions for each vector.  You then add the
%components of the two vectors together, which will give you the unit
%vector description of the total vector.  You then calculate the ratio of
%the component magnitudes in the base vector, and also calculate the ratio
%of the component magnitubes in the combined vector, and ask if the base
%ratio is greater than the combined ratio.  If it is, then the second 
%vector is left or above the base vector, but it is not then the second
%vector is right or below the base vector.

v1 = first_mg-ori_mg;       %the magnitudes of the xy components of the first vector.
v2 = second_mg-ori_mg;      %the magnitudes of the xy components of the second vector.
v_sum = v1+v2;              %add the vectors to create a total vector
v1_ratio = v1(1,:)./v1(2,:); %get the ratio of the two magnitudes for the base vector.
v_sum_ratio = v_sum(1,:)./v_sum(2,:);    %get the ratio of the magnitudes in the combined vector.
a_pos = v1_ratio-v_sum_ratio;       %to get the angle position first subtract the two ratios
a_pos(a_pos>=0) = 1;            %if the combined ratio is smaller than the base ratio, the angle is left of the vector
a_pos(a_pos<0) = -1;            %if the combined ratio is larger than the base ration,the angle is right of the vector